/*******************************************************************************
 *  Copyright 2025 Gaudio Lab, Inc.
 *  All rights reserved.
 *  https://www.gaudiolab.com
 ******************************************************************************/

/**
 * Base error class for Just Voice SDK.
 */
export declare class JVError extends Error {
}
/**
 * Error thrown when the SDK instance has not been created.
 */
export declare class JVNotCreatedError extends JVError {
}
/**
 * Error thrown when the SDK instance has not been initialized.
 */
export declare class JVNotInitializedError extends JVError {
}
/**
 * Error thrown when the SDK instance has already been created.
 */
export declare class JVAlreadyCreatedError extends JVError {
}
/**
 * Error thrown when the SDK instance has already been initialized.
 */
export declare class JVAlreadyInitializedError extends JVError {
}
/**
 * Error thrown when the SDK instance has a null pointer.
 */
export declare class JVNullExceptionError extends JVError {
}
/**
 * Error thrown when the SDK instance has failed to allocate memory.
 */
export declare class JVAllocationFailedError extends JVError {
}
/**
 * Error thrown when the SDK instance has an unsupported number of input channels.
 */
export declare class JVNotSupportedNumInputChannelsError extends JVError {
}
/**
 * Error thrown when the SDK instance has an unsupported number of output channels.
 */
export declare class JVNotSupportedNumOutputChannelsError extends JVError {
}
/**
 * Error thrown when the SDK instance has an unsupported sample rate.
 */
export declare class JVNotSupportedSampleRateError extends JVError {
}
/**
 * Error thrown when the SDK instance has an unsupported samples per block.
 */
export declare class JVNotSupportedSamplesPerBlockError extends JVError {
}
/**
 * Error thrown when the SDK instance has an unsupported intensity.
 */
export declare class JVNotSupportedIntensityError extends JVError {
}

/**
 * Configuration parameters for Just Voice SDK.
 */
export declare class JVParameters {
    /**
     * Level of noise reduction.
     * Value range: 0.0(no noise is removed) ~ 1.0(noise is completely removed)
     */
    noiseReductionIntensity: number;
}

/**
 * Configuration for initializing the JVAudioProcessor.
 */
type JVAudioProcessorConfig = {
    /**
     * URL to the WebAssembly Just Voice SDKmodule.
     * Must be a valid URL pointing to the jv_wasm.js file.
     */
    moduleUrl: string;
};
/**
 * Main class for audio processing in Just Voice SDK.
 * Handles audio input/output and noise reduction.
 */
export declare class JVAudioProcessor {
    /** URL to the Just Voice SDK WebAssembly module. */
    moduleUrl?: string;
    /** Media element to connect to the audio processor. */
    mediaElement?: HTMLMediaElement;
    /** Audio context for the audio processor. */
    audioContext?: AudioContext;
    /** Indicates if the module is loaded. */
    isModuleLoaded: boolean;
    /** Indicates if the setup method has been called. */
    isSetupCalled: boolean;
    /**
     * Load the jv_wasm module.
     * @param config - The configuration for the jv_wasm module.
     * @returns A promise that resolves when the module is loaded.
     */
    loadModule(config: JVAudioProcessorConfig): Promise<void>;
    /**
     * Connect a media element to the audio processor.
     * Note: "connectMediaElement" and "connectAudioNode" cannot be used together.
     * @param mediaElement - The media element to connect.
     * @returns A promise that resolves when the connection is complete.
     */
    connectMediaElement(mediaElement: HTMLMediaElement): Promise<void>;
    /**
     * Connect an audio node to the audio processor.
     * Note: "connectMediaElement" and "connectAudioNode" cannot be used together.
     * @param audioNode - The audio node to connect.
     * @returns A promise that resolves when the connection is complete.
     */
    connectAudioNode(audioNode: AudioNode): Promise<void>;
    /**
     * Get the version of the Just Voice SDK.
     * @returns A promise that resolves to the version of the Just Voice SDK.
     */
    getVersion(): Promise<string>;
    /**
     * Get the latency of the Just Voice SDK.
     * @returns A promise that resolves to the latency of the Just Voice SDK.
     */
    getLatency(): Promise<number>;
    /**
     * Destroy the Just Voice SDK.
     * @returns A promise that resolves when the Just Voice SDK is destroyed.
     */
    destroy(): Promise<void>;
    /**
     * Setup the Just Voice SDK.
     * @param params - The parameters for the Just Voice SDK.
     * @returns A promise that resolves when the Just Voice SDK is setup.
     */
    setup(params: JVParameters): Promise<void>;
    /**
     * Update the Just Voice SDK.
     * @param params - The parameters for the Just Voice SDK.
     * @returns A promise that resolves when the Just Voice SDK is updated.
     */
    update(params: JVParameters): Promise<void>;
    /**
     * Get a parameter value from the Just Voice SDK. (Optional)
     * @param key - The parameter key to retrieve.
     * @returns A promise that resolves to the parameter value.
     */
    getParameter(key: string): Promise<unknown>;
}
export {};
