/*******************************************************************************
 *  Copyright 2025 Gaudio Lab, Inc.
 *  All rights reserved.
 *  http://gaudiolab.com
 ******************************************************************************/

// comment out import JV when using "UMD" module.
import * as JV from './just_voice.esm.js'

//=== Context for example
var exampleCtx = {}
exampleCtx.isPlay = false
exampleCtx.isAutoPlay = false
exampleCtx.streamingMode = undefined
exampleCtx.micStream = undefined
exampleCtx.playBtn = $('#play')
exampleCtx.media = $('#media')

//--- Constants
var exampleConst = Object.freeze({
  kStreamPD: 'pd',
  kStreamAudio: 'audio',
  kStreamMic: 'mic',

  kPlaylist: Object.freeze({
    pd: {
      url: 'https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/SubaruOutbackOnStreetAndDirt.mp4'
    },
    audio: {
      url: 'https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/SubaruOutbackOnStreetAndDirt.mp4'
    }
  })
});

//=== JV
var jv = new JV.JVAudioProcessor();

var jvParams = new JV.JVParameters({
  noiseReductionIntensity: 1  // Configure how strongly to denoise. (Support: 0.0 ~ 1.0)
});

//=== JV functions
function JVSetup(jvParams) {
  jv.setup(jvParams).then(() => {
    // success, do nothing
  })
  .catch((err) => {
    alert("JV setup fail: " + err);
  });
}

function JVUpdate(jvParams) {
  jv.update(jvParams).then(() => {
    // success, do nothing
  })
  .catch((err) => {
    alert("JV update fail: " + err);
  });
}

function JVDestroy() {
  jv.destroy().then(() => {
    // success, do nothing
  })
  .catch((err) => {
    alert("JV update fail: " + err);
  });
}

// Returns the process latency(ms)
function JVGetLatency() {
  jv.getLatency()
    .then(function (latency) {
      $('#latency-value').html(latency.toFixed(3));
    })
    .catch((err) => {
      alert("JV get latency fail: " + err);
    });
}

function JVCreate(audioSource) {
  _createJV(audioSource);

  function _createJV(audioSource) {
    if (!jv.isModuleLoaded) {
      alert('library not loaded!');
      return;
    }

    // Refer to the click function below as well.
    // Use below when the input source is mediaElement (video or audio).
    // Not supported using microphone input because it is not a mediaElement.
    // jv.connectMediaElement(audioSource);

    // Use below when the input source is audioNode
    jv.connectAudioNode(audioSource);
    
    jv.isSetupCalled ? JVUpdate(jvParams) : JVSetup(jvParams);
  }
}

//=== Player functions
async function setPlay() {
  if (!exampleCtx.isPlay) {
    if (exampleCtx.streamingMode == exampleConst.kStreamMic) {
      try {
        // get mic stream
        exampleCtx.micStream = await navigator.mediaDevices.getUserMedia({ audio: true });
    
        // mic support only audioNode
        const micSource = jv.audioContext.createMediaStreamSource(exampleCtx.micStream);
        JVCreate(micSource);

        $('#latency-value').html('-');
      } catch (err) {
        alert("Fail to MIC access: " + err);
      }
    } else {
      exampleCtx.media[0].play();
    }

    exampleCtx.playBtn.html("pause");
    exampleCtx.isPlay = true;
  }
}

function setPause() {
  if (exampleCtx.isPlay) {
    if (exampleCtx.streamingMode == exampleConst.kStreamMic) {
      if (exampleCtx.micStream) {
        exampleCtx.micStream.getTracks().forEach(track => track.stop());
        exampleCtx.micStream = undefined;

        JVGetLatency();
      }
    } else {
      exampleCtx.media[0].pause();
    }
    exampleCtx.playBtn.html("play");
    exampleCtx.isPlay = false;
  }
}

function changeStreamingMode(newStreamingMode) {
  // remove media tag(video or audio)
  $('#media').remove();

  if(newStreamingMode == exampleConst.kStreamMic) {
    // do nothing
  } else {
    switch (newStreamingMode) {
      case exampleConst.kStreamPD:
        $('#media-tag').append('<video id="media" playsinline></video>');
        break;
      case exampleConst.kStreamAudio:
      default:
        $('#media-tag').append('<audio id="media" playsinline></audio>');
        break;
    }
  
    exampleCtx.media = $('#media');
    exampleCtx.media[0].crossOrigin = "anonymous";

    exampleCtx.media[0].addEventListener('play', function() {
      $('#latency-value').html('-');
    });
    exampleCtx.media[0].addEventListener('pause', function() {
      JVGetLatency();
    });
  }
  exampleCtx.streamingMode = newStreamingMode;
}

//=== INIT
(function init() {
  function initJV() {
    jv.loadModule({moduleUrl: './jv_wasm.js'})
      .then(function () {
        return jv.getVersion();
      })
      .then(function (version) {
        $('#jv-version').html('Version: ' + version);
      })
      .catch((err) => {
        alert("JV loadModule fail: " + err);
      });
  }

  function registerEventHandlers() {
    // JV Destructor
    $(window).on("beforeunload", function () {
      console.log("[JV] Call to destroy() before unload successfully.");
      JVDestroy();
    });

    // JV noise reduction intensity
    $('#noise-reduction-intensity-slider').change(function () {
      jvParams.noiseReductionIntensity = parseFloat($(this).val());
      $('#noise-reduction-intensity-value').html(jvParams.noiseReductionIntensity);

      if (jv.isSetupCalled) {
        JVUpdate(jvParams);
      }
    });

    // Player
    $('#pd').click(function () {
      setPause();
      changeStreamingMode(exampleConst.kStreamPD);

      // Refer to the _createJV function above as well.
      // Use below when the input source is video mediaElement
      // const inputSource = exampleCtx.media[0];
      // Use below when the input source is audioNode
      const inputSource = jv.audioContext.createMediaElementSource(exampleCtx.media[0]);
      JVCreate(inputSource);
      
      exampleCtx.media[0].src = ($('#url').val().length) ? $('#url').val() : exampleConst.kPlaylist.pd.url;
      exampleCtx.media[0].type = undefined
      exampleCtx.media[0].load();
      if (exampleCtx.isAutoPlay) {
        setPlay();
      }
    });
    $('#audio').click(function () {
      setPause();
      changeStreamingMode(exampleConst.kStreamAudio);

      // Refer to the _createJV function above as well.
      // Use below when the input source is audio mediaElement
      // const inputSource = exampleCtx.media[0];
      // Use below when the input source is audioNode
      const inputSource = jv.audioContext.createMediaElementSource(exampleCtx.media[0]);
      JVCreate(inputSource);
      
      exampleCtx.media[0].src = ($('#url').val().length) ? $('#url').val() : exampleConst.kPlaylist.audio.url;
      exampleCtx.media[0].type = undefined;
      exampleCtx.media[0].load();
      if (exampleCtx.isAutoPlay) {
        setPlay();
      }
    });
    $('#mic').click(function () {
      setPause();
      changeStreamingMode(exampleConst.kStreamMic);

      if (exampleCtx.isAutoPlay) {
        setPlay();
      }
    });

    $("#play").click(function () {
      exampleCtx.isPlay ? setPause() : setPlay();
    });
    $("#autoplay").change(function () {
      exampleCtx.isAutoPlay = $(this).is(':checked');
    });
  }

  function updatePlayer() {
    exampleCtx.media[0].crossOrigin = "anonymous";
  }

  function updateUIs() {
    // Player
    $("#autoplay").prop('checked', exampleCtx.isAutoPlay);
  }

  initJV();
  registerEventHandlers();
  updatePlayer();
  updateUIs();
})();
