# Just Voice WebRTC Integration Guide
본 문서는 Just Voice를 WebRTC에 통합하는 과정을 다룹니다.

## 목차
- [개요](#개요)
- [통합](#통합)
  - [파일 옮기기](#파일-옮기기)
  - [`JustVoiceImpl` 클래스 작성](#justvoiceimpl-클래스-작성)
  - [`AudioProcessingImpl` 클래스에 `JustVoiceImpl` 클래스를 *Submodule*로 등록](#audioprocessingimpl-클래스에-justvoiceimpl-클래스를-submodule로-등록)
  - [실제 처리 로직을 `ProcessCaptureStreamLocked` 함수에 통합](#실제-처리-로직을-processcapturestreamlocked-함수에-통합)
  - [빌드 스크립트를 수정해서 라이브러리를 링크](#빌드-스크립트를-수정해서-라이브러리를-링크)
- [검증](#검증)

## 개요
WebRTC의 신호 처리는 *AudioProcessingModule*을 통해 이루어집니다.
본 문서의 목표는 Just Voice SDK를 *AudioProcessingModule*의 *Submodule*로 등록해서 신호 처리 파이프라인에 추가하는 것입니다.
이는 다음의 과정으로 이루어집니다.

1. 인스턴스를 관리하기 위한 `JustVoiceImpl` 클래스 작성
2. `AudioProcessingImpl` 클래스에 `JustVoiceImpl` 클래스를 *Submodule*로 등록
3. 실제 처리 로직을 `ProcessCaptureStreamLocked` 함수에 통합
4. 빌드 스크립트를 수정해서 라이브러리를 링크

본 문서의 내용은 다음의 환경에서 테스트 되었습니다.

<table>
<thead>
  <tr>
    <th>target_os</th>
    <th>target_cpu</th>
    <th>WebRTC Commit ID</th>
  </tr>
</thead>
<tbody>
  <tr>
    <td>win</td>
    <td>x64</td>
    <td rowspan="3">ecdedac</td>
  </tr>
  <tr>
    <td rowspan="2">mac</td>
    <td>x64</td>
  </tr>
  <tr>
    <td>arm64</td>
  </tr>
  <tr>
    <td rowspan="2">android</td>
    <td>arm</td>
    <td rowspan="3">c71bfcc</td>
  </tr>
  <tr>
    <td>arm64</td>
  </tr>
  <tr>
    <td>ios</td>
    <td>arm64</td>
  </tr>
</tbody>
</table>

## 통합
기존 코드에 추가 및 수정한 부분은 `JUST VOICE BEGIN` 및 `JUST VOICE END` 주석을 달아 놓았습니다. WebRTC 버전에 따라 구체적인 코드는 차이가 있을 수 있습니다.

### 파일 옮기기
* `just_voice_sdk` 폴더를 WebRTC 프로젝트의 modules/audio_processing로 옮깁니다. WebRTC 프로젝트는 Chromium 프로젝트 기준, third_party/webrtc에 위치합니다.

### `JustVoiceImpl` 클래스 작성
* modules/audio_processing/just_voice_impl.h 생성
```cpp
#ifndef MODULES_AUDIO_PROCESSING_JV_IMPL_H_
#define MODULES_AUDIO_PROCESSING_JV_IMPL_H_

#include <cstdint>
#include <cstring>

#include "just_voice_sdk/public/just_voice.h"

namespace webrtc {
class AudioBuffer;

class JustVoiceImpl {
private:
  size_t                num_channels_;
  just_voice_handle_t** handle_      ;
  float               * input_       ;
  float               * output_      ;

public:
  JustVoiceImpl(int32_t sample_rate_hz, size_t num_channels) noexcept;
  ~JustVoiceImpl() noexcept;

  JustVoiceImpl(JustVoiceImpl const&) noexcept = delete;
  auto operator=(JustVoiceImpl const&) noexcept -> JustVoiceImpl& = delete;

  auto Process(AudioBuffer* audio) noexcept -> void;
};
}

#endif // MODULES_AUDIO_PROCESSING_JV_IMPL_H_
```

* modules/audio_processing/just_voice_impl.cc 생성
```cpp
#include "modules/audio_processing/just_voice_impl.h"

#include "api/array_view.h"
#include "modules/audio_processing/audio_buffer.h"
#include "rtc_base/checks.h"
#include "rtc_base/logging.h"

namespace webrtc {
JustVoiceImpl::JustVoiceImpl(int32_t sample_rate_hz, size_t num_channels) noexcept:
  num_channels_(num_channels                                      ),
  handle_      (new just_voice_handle_t*[ num_channels         ]()),
  input_       (new float            [ sample_rate_hz / 100 ]     ),
  output_      (new float            [ sample_rate_hz / 100 ]     ) {
  auto const config = just_voice_config_t {
    1                                            , // numInputChannels
    1                                            , // numOutputChannels
    static_cast< uint32_t >(sample_rate_hz      ), // sampleRate
    static_cast< uint32_t >(sample_rate_hz / 100)  // samplesPerBlock
  };

  auto const params = just_voice_params_t {
    1.f                                            // noiseReductionIntensity
  };

  for (auto k = (size_t)0; k < num_channels_; ++k) {
    JV_CREATE(&handle_[ k ]);
    JV_SETUP(handle_[ k ], &config, &params);
  }
}

JustVoiceImpl::~JustVoiceImpl() noexcept {
  for (auto k = (size_t)0; k < num_channels_; ++k)
    JV_DESTROY(&handle_[ k ]);

  delete[] handle_;
  delete[] input_ ;
  delete[] output_;
}

auto JustVoiceImpl::Process(AudioBuffer* audio) noexcept -> void {
  RTC_DCHECK(audio);
  RTC_DCHECK_EQ(num_channels_, audio->num_channels());

  auto const samples_per_block = audio->num_frames();

  for (auto k = (size_t)0; k < num_channels_; ++k) {
    auto const in  = rtc::ArrayView< float const >(&audio->channels()[ k ][ 0 ], samples_per_block);
    auto const out = rtc::ArrayView< float       >(&audio->channels()[ k ][ 0 ], samples_per_block);

    FloatS16ToFloat(in.data()   , samples_per_block, input_                       );
    JV_PROCESS(handle_[ k ], input_           , output_   , samples_per_block);
    FloatToFloatS16(output_     , samples_per_block, out.data()                   );
  }
}
}
```

### `AudioProcessingImpl` 클래스에 `JustVoiceImpl` 클래스를 *Submodule*로 등록
* modules/audio_processing/include/audio_processing.h 수정

```cpp
class RTC_EXPORT AudioProcessing : public rtc::RefCountInterface {

...

      struct FixedDigital {
        float gain_db = 0.0f;
      } fixed_digital;
    } gain_controller2;

    // JUST VOICE BEGIN
    struct JustVoice {
      bool enabled = true;
    } just_voice;
    // JUST VOICE END

    std::string ToString() const;
  };

...
```

* modules/audio_processing/include/audio_processing.cc 수정
```cpp
...

          << gain_controller2.adaptive_digital.max_output_noise_level_dbfs
          << " }, input_volume_control : { enabled "
          << gain_controller2.input_volume_controller.enabled << "}"
          // JUST VOICE BEGIN
          << ", just_voice : { enabled "
          << just_voice.enabled << " }}";
          // JUST VOICE END

...
```

### 실제 처리 로직을 `ProcessCaptureStreamLocked` 함수에 통합
* modules/audio_processing/audio_processing_impl.h 수정
```cpp
...

// JUST VOICE BEGIN
#include "modules/audio_processing/just_voice_impl.h"
// JUST VOICE END

...

  class SubmoduleStates {

...

    bool Update(bool high_pass_filter_enabled,
                bool mobile_echo_controller_enabled,
                bool noise_suppressor_enabled,
                bool adaptive_gain_controller_enabled,
                bool gain_controller2_enabled,
                bool voice_activity_detector_enabled,
                bool gain_adjustment_enabled,
                bool echo_controller_enabled,
                bool transient_suppressor_enabled,
                // JUST VOICE BEGIN
                bool just_voice_enabled);
                // JUST VOICE END

...

    bool transient_suppressor_enabled_ = false;
    // JUST VOICE BEGIN
    bool just_voice_enabled_ = false;
    // JUST VOICE END
    bool first_update_ = true;

...

  void InitializeAnalyzer() RTC_EXCLUSIVE_LOCKS_REQUIRED(mutex_capture_);
  // JUST VOICE BEGIN
  void InitializeJustVoice() RTC_EXCLUSIVE_LOCKS_REQUIRED(mutex_capture_);
  // JUST VOICE END
  void InitializePreProcessor() RTC_EXCLUSIVE_LOCKS_REQUIRED(mutex_render_);

...

    std::unique_ptr<TransientSuppressor> transient_suppressor;
    // JUST VOICE BEGIN
    std::unique_ptr<JustVoiceImpl> just_voice;
    // JUST VOICE END
    std::unique_ptr<CaptureLevelsAdjuster> capture_levels_adjuster;

...
```

* modules/audio_processing/audio_processing_impl.cc 수정
```cpp
...

bool AudioProcessingImpl::SubmoduleStates::Update(
    bool high_pass_filter_enabled,
    bool mobile_echo_controller_enabled,
    bool noise_suppressor_enabled,
    bool adaptive_gain_controller_enabled,
    bool gain_controller2_enabled,
    bool voice_activity_detector_enabled,
    bool gain_adjustment_enabled,
    bool echo_controller_enabled,
    bool transient_suppressor_enabled,
    // JUST VOICE BEGIN
    bool just_voice_enabled) {
    // JUST VOICE END
  bool changed = false;
  changed |= (high_pass_filter_enabled != high_pass_filter_enabled_);
  changed |=
      (mobile_echo_controller_enabled != mobile_echo_controller_enabled_);
  changed |= (noise_suppressor_enabled != noise_suppressor_enabled_);
  changed |=
      (adaptive_gain_controller_enabled != adaptive_gain_controller_enabled_);
  changed |= (gain_controller2_enabled != gain_controller2_enabled_);
  changed |=
      (voice_activity_detector_enabled != voice_activity_detector_enabled_);
  changed |= (gain_adjustment_enabled != gain_adjustment_enabled_);
  changed |= (echo_controller_enabled != echo_controller_enabled_);
  changed |= (transient_suppressor_enabled != transient_suppressor_enabled_);
  // JUST VOICE BEGIN
  changed |= (just_voice_enabled != just_voice_enabled_);
  // JUST VOICE END
  if (changed) {
    high_pass_filter_enabled_ = high_pass_filter_enabled;
    mobile_echo_controller_enabled_ = mobile_echo_controller_enabled;
    noise_suppressor_enabled_ = noise_suppressor_enabled;
    adaptive_gain_controller_enabled_ = adaptive_gain_controller_enabled;
    gain_controller2_enabled_ = gain_controller2_enabled;
    voice_activity_detector_enabled_ = voice_activity_detector_enabled;
    gain_adjustment_enabled_ = gain_adjustment_enabled;
    echo_controller_enabled_ = echo_controller_enabled;
    transient_suppressor_enabled_ = transient_suppressor_enabled;
    // JUST VOICE BEGIN
    just_voice_enabled_ = just_voice_enabled;
    // JUST VOICE END
  }

  changed |= first_update_;
  first_update_ = false;
  return changed;
}

...

  InitializeAnalyzer();
  // JUST VOICE BEGIN
  InitializeJustVoice();
  // JUST VOICE END
  InitializePostProcessor();

...

int AudioProcessingImpl::ProcessCaptureStreamLocked() {
  EmptyQueuedRenderAudioLocked();
  HandleCaptureRuntimeSettings();
  DenormalDisabler denormal_disabler;

  // Ensure that not both the AEC and AECM are active at the same time.
  // TODO(peah): Simplify once the public API Enable functions for these
  // are moved to APM.
  RTC_DCHECK_LE(
      !!submodules_.echo_controller + !!submodules_.echo_control_mobile, 1);

  data_dumper_->DumpRaw(
      "applied_input_volume",
      capture_.applied_input_volume.value_or(kUnspecifiedDataDumpInputVolume));

  AudioBuffer* capture_buffer = capture_.capture_audio.get();  // For brevity.
  AudioBuffer* linear_aec_buffer = capture_.linear_aec_output.get();

  // JUST VOICE BEGIN
  if (config_.just_voice.enabled) {
    submodules_.just_voice->Process(capture_buffer);
  }
  // JUST VOICE END

  if (submodules_.high_pass_filter &&
      config_.high_pass_filter.apply_in_full_band &&
      !constants_.enforce_split_band_hpf) {
    submodules_.high_pass_filter->Process(capture_buffer,
                                          /*use_split_band_data=*/false);
  }

...

bool AudioProcessingImpl::UpdateActiveSubmoduleStates() {
  return submodule_states_.Update(
      config_.high_pass_filter.enabled, !!submodules_.echo_control_mobile,
      !!submodules_.noise_suppressor, !!submodules_.gain_control,
      !!submodules_.gain_controller2, !!submodules_.voice_activity_detector,
      config_.pre_amplifier.enabled || config_.capture_level_adjustment.enabled,
      capture_nonlocked_.echo_controller_enabled,
      !!submodules_.transient_suppressor,
      // JUST VOICE BEGIN
      config_.just_voice.enabled);
      // JUST VOICE END
}

...

// JUST VOICE BEGIN
void AudioProcessingImpl::InitializeJustVoice() {
  submodules_.just_voice.reset();
  if (config_.just_voice.enabled) {
    submodules_.just_voice = std::make_unique<JustVoiceImpl>(proc_sample_rate_hz(), num_proc_channels());
  }
}
// JUST VOICE END

...
```

### 빌드 스크립트를 수정해서 라이브러리를 링크
* modules/audio_processing/BUILD.gn 수정
```
...

rtc_library("audio_buffer") {
  ...
}

# JUST VOICE BEGIN
rtc_library("just_voice_impl") {
  visibility = [ "*" ]

  sources = [
    "just_voice_impl.cc",
    "just_voice_impl.h",
  ]

  if (is_mac) {
    if (target_cpu == "arm64") {
      libs = [ "just_voice_sdk/lib/macosx/arm64/static/Release/libjust_voice.a" ]
    }
    else if (target_cpu == "x64") {
      libs = [ "just_voice_sdk/lib/macosx/x86_64/static/Release/libjust_voice.a" ]
    }
    else {
      assert(false, "Unknown target_cpu for macOS!!")
    }
  }
  else if (is_win) {
    if (is_debug) {
      if (target_cpu == "x64") {
        libs = [ "just_voice_sdk/lib/windows/v143/MT/x86_64/static/Debug/just_voice.lib" ]
      }
      else if  (target_cpu == "x86") {
        libs = [ "just_voice_sdk/lib/windows/v143/MT/x86/static/Debug/just_voice.lib" ]
      }
      else {
        assert(false, "Unknown target_cpu for windows!!")
      }
    }
    else {
      if (target_cpu == "x64") {
        libs = [ "just_voice_sdk/lib/windows/v143/MT/x86_64/static/Release/just_voice.lib" ]
      }
      else if  (target_cpu == "x86") {
        libs = [ "just_voice_sdk/lib/windows/v143/MT/x86/static/Release/just_voice.lib" ]
      }
      else {
        assert(false, "Unknown target_cpu for windows!!")
      }
    }
  }
  else if (is_android) {
    if (target_cpu == "arm") {
      libs = [ "just_voice_sdk/lib/android/armeabi-v7a-neon/static/Release/libjust_voice.a" ]
    }
    else if (target_cpu == "arm64") {
      libs = [ "just_voice_sdk/lib/android/arm64-v8a/static/Release/libjust_voice.a" ]
    }
    else if (target_cpu == "x86") {
      libs = [ "just_voice_sdk/lib/android/x86/static/Release/libjust_voice.a" ]
    }
    else if (target_cpu == "x64") {
      libs = [ "just_voice_sdk/lib/android/x86_64/static/Release/libjust_voice.a" ]
    }
    else {
      assert(false, "Unknown target_cpu for Android!!")
    }
  }
  else if (is_ios) {
    if (target_cpu == "arm64") {
      libs = [ "just_voice_sdk/lib/iphoneos/arm64/static/Release/libjust_voice.a" ]
    }
    else if (target_cpu == "x64") {
      libs = [ "just_voice_sdk/lib/iphonesimulator/x86_64/static/Release/libjust_voice.a" ]
    }
    else {
      assert(false, "Unknown target_cpu for iOS!!")
    }
  }
  else {
    assert(false, "Unknown OS!!")
  }

  include_dirs = [
    "just_voice_sdk/include"
  ]

  deps = [
    ":audio_buffer",
    "../../api:array_view",
    "../../rtc_base:checks",
    "../../rtc_base:logging",
  ]
}
# JUST VOICE END

rtc_library("high_pass_filter") {
  ...
}

...

rtc_library("audio_processing") {
  visibility = [ "*" ]
  configs += [ ":apm_debug_dump" ]
  sources = [
    "audio_processing_builder_impl.cc",
    "audio_processing_impl.cc",
    "audio_processing_impl.h",
    "echo_control_mobile_impl.cc",
    "echo_control_mobile_impl.h",
    "gain_control_impl.cc",
    "gain_control_impl.h",
    "render_queue_item_verifier.h",
  ]

  defines = []
  deps = [
    ":aec_dump_interface",
    ":api",
    ":apm_logging",
    ":audio_buffer",
    ":audio_frame_proxies",
    ":audio_frame_view",
    ":audio_processing_statistics",
    ":gain_controller2",
    # JUST VOICE BEGIN
    ":just_voice_impl",
    # JUST VOICE END
    ":high_pass_filter",

...

    rtc_library("audio_processing_unittests") {
      testonly = true

      configs += [ ":apm_debug_dump" ]
      sources = [
        "audio_buffer_unittest.cc",
        "audio_frame_view_unittest.cc",
        "echo_control_mobile_unittest.cc",
        "gain_controller2_unittest.cc",
        "splitting_filter_unittest.cc",
        "test/fake_recording_device_unittest.cc",
      ]

      deps = [
        ":analog_mic_simulation",
        ":api",
        ":apm_logging",
        ":audio_buffer",
        ":audio_frame_view",
        ":audio_processing",
        ":audioproc_test_utils",
        ":gain_controller2",
        # JUST VOICE BEGIN
        ":just_voice_impl",
        # JUST VOICE END
        ":high_pass_filter",

...
```

## 검증
WebRTC를 빌드하면, *AudioProcessingModule*의 CLI인 `audioproc_f` 파일이 생성됩니다.
이를 활용하여 Just Voice SDK가 정상적으로 통합되었는지 확인할 수 있습니다.
사용 방법은 다음과 같습니다.

```sh
./out/Default/audioproc_f -i INPUT.wav -o OUTPUT.wav
```
