# Just Voice API Reference
본 문서는 Just Voice C API의 전반적인 사용법을 다룹니다.



## Table of Contents
- [Just Voice API Reference](#just-voice-api-reference)
  - [Table of Contents](#table-of-contents)
  - [API Life Cycle](#api-life-cycle)
    - [Note](#note)
  - [API Description](#api-description)
    - [Type](#type)
      - [`just_voice_handle_t` (Type Alias)](#just_voice_handle_t-type-alias)
      - [`just_voice_config_t` (Struct)](#just_voice_config_t-struct)
      - [`just_voice_params_t` (Struct)](#just_voice_params_t-struct)
      - [Error Code (Enum)](#error-code-enum)
    - [함수](#함수)
      - [`JV_CREATE`](#jv_create)
        - [설명](#설명)
        - [사용 예시](#사용-예시)
      - [`JV_SETUP`](#jv_setup)
        - [설명](#설명-1)
        - [사용 예시](#사용-예시-1)
      - [`JV_UPDATE`](#jv_update)
        - [설명](#설명-2)
        - [사용 예시](#사용-예시-2)
      - [`JV_PROCESS`](#jv_process)
        - [설명](#설명-3)
        - [사용 예시](#사용-예시-3)
      - [`JV_DESTROY`](#jv_destroy)
        - [설명](#설명-4)
        - [사용 예시](#사용-예시-4)
      - [`JV_GET_LATENCY`](#jv_get_latency)
        - [설명](#설명-5)
        - [사용 예시](#사용-예시-5)
      - [`JV_VERSION`](#jv_version)
        - [설명](#설명-6)
        - [사용 예시](#사용-예시-6)



## API Life Cycle

![API Lifecycle](resources/lifecycle.svg)

### Note
- 동작에 필요한 초기 설정값은 `just_voice_config_t`와 `just_voice_params_t`에 입력한 다음, `JV_SETUP`을 통해 설정합니다.
- INITIALIZED 이후에는 `just_voice_params_t`의 항목만 `JV_UPDATE`를 통해 업데이트 할 수 있습니다.



## API Description

### Type
#### `just_voice_handle_t` (Type Alias)
```c
typedef void* just_voice_handle_t;
```

> `just_voice_handle_t` 생성한 Just Voice 인스턴스의 포인터를 저장합니다.



#### `just_voice_config_t` (Struct)
| Field | Type | Value | Description |
| ------------------- | ---------- | - | - |
| `numInputChannels`  | `uint32_t` | 1 이상 | 입력 채널의 수 |
| `numOutputChannels` | `uint32_t` | `numInputChannels`와 같은 값 | 출력 채널의 수 |
| `sampleRate`        | `uint32_t` | 8000, 11025, 16000, 22050, 24000, 32000, 44100, 48000, 64000, 88200, 96000, 176400, 192000 | 입출력의 샘플 레이트 |
| `samplesPerBlock`   | `uint32_t` | 0, 1 ~ 32768 | 입출력 블록 당 샘플 수 (유동적인 경우 0으로 설정) |

> `just_voice_config_t`는 `JV_SETUP`을 통해 설정한 이후 업데이트 할 수 없습니다.



#### `just_voice_params_t` (Struct)
| Field | Type | Value | Description |
| ------------------------- | ------- | --------- | - |
| `noiseReductionIntensity` | `float` | 0.f ~ 1.f | 잡음 제거 강도를 설정합니다. (0인 경우 입력 그대로 출력, 1인 경우 최대로 제거) |

> `just_voice_params_t`는 `JV_SETUP`을 통해 설정한 후, `JV_UPDATE`을 통해 프로세싱 중에 업데이트 할 수 있습니다.



#### Error Code (Enum)
| 이름 | 값 | 설명 |
| - | - | - |
| `JV_SUCCESS`                           | 0 | API가 올바르게 동작했습니다. |
| `JV_NOT_CREATED` | 1 | 인스턴스가 생성되지 않아 작업을 수행할 수 없습니다. |
| `JV_NOT_INITIALIZED`                   | 2 | 인스턴스가 초기화되지 않아 작업을 수행할 수 없습니다. |
| `JV_ALREADY_CREATED` | 3 | 인스턴스가 이미 생성되어 작업을 수행할 수 없습니다. |
| `JV_ALREADY_INITIALIZED`               | 4 | 인스턴스가 이미 초기화되어 작업을 수행할 수 없습니다. |
| `JV_NULL_EXCEPTION`                    | 5 | 포인터의 값이 `NULL`이라 작업을 수행할 수 없습니다. |
| `JV_ALLOCATION_FAILED`                 | 6 | 내부 버퍼 할당에 실패하여 작업을 수행할 수 없습니다. |
| `JV_NOT_SUPPORTED_NUM_INPUT_CHANNELS`  | 7 | 지원하지 않는 입력 채널 수를 입력하여 작업을 수행할 수 없습니다. |
| `JV_NOT_SUPPORTED_NUM_OUTPUT_CHANNELS` | 8 | 지원하지 않는 출력 채널 수를 입력하여 작업을 수행할 수 없습니다. |
| `JV_NOT_SUPPORTED_SAMPLE_RATE`         | 9 | 지원하지 않는 샘플 레이트를 입력하여 작업을 수행할 수 없습니다. |
| `JV_NOT_SUPPORTED_SAMPLES_PER_BLOCK`   | 10 | 지원하지 않는 블록 당 샘플 수를 입력하여 작업을 수행할 수 없습니다. |
| `JV_NOT_SUPPORTED_INTENSITY`           | 11 | 지원하지 않는 노이즈 제거 강도를 입력하여 작업을 수행할 수 없습니다. |
> 에러 코드는 API 함수들의 반환값으로 사용됩니다.




### 함수
#### `JV_CREATE`
```c
int32_t JV_CREATE(just_voice_handle_t** handle);
```

##### 설명
인스턴스를 생성합니다. 사용 후, 자원의 해제를 위해 `JV_DESTROY`를 호출해야 합니다.

| 이름 | 설명 |
| - | - |
| `handle` | 새로 인스턴스를 생성할 포인터를 지정합니다. 포인팅 주소는 `NULL`이어야 합니다. |

##### 사용 예시
```c
just_voice_handle_t* handle = NULL;
int32_t const create_result = JV_CREATE(&handle);

if (create_result != JV_SUCCESS) { ... }
```



#### `JV_SETUP`
```c
int32_t JV_SETUP(just_voice_handle_t* handle, just_voice_config_t const* config, just_voice_params_t const* params);
```

##### 설명
`just_voice_config_t`와 `just_voice_params_t`를 받아, 렌더링에 필요한 정보를 인스턴스에 설정합니다.

| 이름 | 설명 |
| - | - |
| `handle` | 정보를 설정할 인스턴스를 지정합니다. |
| `config` | 설정값을 입력합니다.(이후 업데이트 불가) |
| `params` | 설정값을 입력합니다.(이후 업데이트 가능) |

##### 사용 예시
```c
just_voice_handle_t* handle = NULL;
just_voice_config_t const config = { 1, 1, 48000, 480 };
just_voice_params_t const params = { 1.f };

...

int32_t const setup_result = JV_SETUP(handle, &config, &param);

if (setup_result != JV_SUCCESS) { ... }
```



#### `JV_UPDATE`
```c
int32_t JV_UPDATE(just_voice_handle_t* handle, just_voice_params_t const* params);
```

##### 설명
`just_voice_params_t`를 받아, 렌더링에 필요한 정보를 인스턴스에 설정합니다.
`JV_SETUP`와는 다르게, 여러 번 호출하여 그 때 그 때 다른 설정으로 렌더링할 수 있도록 합니다.

| 이름 | 설명 |
| - | - |
| `handle` | 정보를 설정할 인스턴스를 지정합니다. |
| `params` | 업데이트할 값을 입력합니다. |

##### 사용 예시
```c
just_voice_handle_t* handle = NULL;
just_voice_params_t const params = { 1.f };

...

int32_t const update_result = JV_UPDATE(handle, &param);

if (update_result != JV_SUCCESS) { ... }
```



#### `JV_PROCESS`
```c
int32_t JV_PROCESS(just_voice_handle_t* handle, float const* in, float* out, uint32_t length);
```

##### 설명
입력 버퍼의 오디오를 프로세싱하여 출력 버퍼에 넣습니다.
> `samplesPerBlock`을 0으로 설정한 경우, 임의의 입력 버퍼 길이를 입력할 수 있습니다.

| 이름 | 설명 |
| - | - |
| `handle` | 프로세싱을 수행할 인스턴스를 지정합니다. |
| `input` | 입력 오디오 정보를 담고 있는 버퍼를 지정합니다. |
| `output` | 출력 오디오 정보를 담을 버퍼를 지정합니다. |
| `length` | 채널당 입출력 샘플 수를 지정합니다. |

##### 사용 예시
```c
just_voice_handle_t* handle = NULL;
float const* input_buffer = ...;
float* output_buffer = ...;
uint32_t length = ...;

...

int32_t const process_result = JV_PROCESS(handle, input_buffer, output_buffer, length);

if (process_result != JV_SUCCESS) { ... }
```



#### `JV_DESTROY`
```c
int32_t JV_DESTROY(just_voice_handle_t** handle);
```

##### 설명
인스턴스를 제거하고 할당한 자원을 해제합니다.

| 이름 | 설명 |
| - | - |
| `handle` | 제거할 인스턴스의 포인터를 지정합니다. 제거 후, `NULL` 값이 설정됩니다. |

##### 사용 예시
```c
just_voice_handle_t* handle = NULL;

...

int32_t destroy_result = JV_DESTROY(&handle);

if (destroy_result != JV_SUCCESS) { ... }
```


#### `JV_GET_LATENCY`
```c
int32_t JV_GET_LATENCY(just_voice_handle_t const* handle, float* latency);
```

##### 설명
알고리즘 레이턴시를 ms 단위로 계산하여 제공합니다.

| 이름 | 설명 |
| - | - |
| `handle` | 렌더링 정보가 담긴 인스턴스를 지정합니다. |
| `latency` | 레이턴시 정보를 받을 변수를 지정합니다. |

##### 사용 예시
```c
just_voice_handle_t* handle = NULL;
float* latency = 0.f;

...

int32_t const latency_result = JV_GET_LATENCY(handle, &latency);

if (latency_result != JV_SUCCESS) { ... }
```



#### `JV_VERSION`
```c
int32_t JV_GET_VERSION(char const** version);
```

##### 설명
SDK의 버전 정보를 제공합니다.

| 이름 | 설명 |
| - | - |
| `version` | 버전 정보를 받을 변수를 지정합니다. |

##### 사용 예시
```c
char const* version;
int32_t const version_result = JV_GET_VERSION(&version);

if (version_result != JV_SUCCESS) { ... }
```
