# Just Voice API Reference
Introducing Just Voice’s native C API.



## Table of Contents
- [Just Voice API Reference](#just-voice-api-reference)
  - [Table of Contents](#table-of-contents)
  - [API Life Cycle](#api-life-cycle)
    - [Note](#note)
  - [API Description](#api-description)
    - [Type](#type)
      - [`just_voice_handle_t` (Type Alias)](#just_voice_handle_t-type-alias)
      - [`just_voice_config_t` (Struct)](#just_voice_config_t-struct)
      - [`just_voice_params_t` (Struct)](#just_voice_params_t-struct)
      - [Error Codes (Enum)](#error-codes-enum)
    - [Function](#function)
      - [`JV_CREATE`](#jv_create)
        - [Description](#description)
        - [Example](#example)
      - [`JV_SETUP`](#jv_setup)
        - [Description](#description-1)
        - [Example](#example-1)
      - [`JV_UPDATE`](#jv_update)
        - [Description](#description-2)
        - [Example](#example-2)
      - [`JV_PROCESS`](#jv_process)
        - [Description](#description-3)
        - [Example](#example-3)
      - [`JV_DESTROY`](#jv_destroy)
        - [Description](#description-4)
        - [Example](#example-4)
      - [`JV_GET_LATENCY`](#jv_get_latency)
        - [Description](#description-5)
        - [Example](#example-5)
      - [`JV_VERSION`](#jv_version)
        - [Description](#description-6)
        - [Example](#example-6)



## API Life Cycle
![API Lifecycle](resources/lifecycle.svg)

### Note
- The initial settings required for processing are entered into `just_voice_config_t` & `just_voice_params_t` and then set through `JV_SETUP`.
- After INITIALIZED, only `just_voice_params_t` can be updated through `JV_UPDATE`.



## API Description

### Type
#### `just_voice_handle_t` (Type Alias)
```c
typedef void* just_voice_handle_t;
```

> `just_voice_handle_t` stores a pointer to the JustVoice instance.



#### `just_voice_config_t` (Struct)
| Field | Type | Value | Description |
| ------------------- | ---------- | - | - |
| `numInputChannels`  | `uint32_t` | 1 or more | Number of input channels |
| `numOutputChannels` | `uint32_t` | Same value as `numInputChannels` | Number of output channels |
| `sampleRate`        | `uint32_t` | 8000, 11025, 16000, 22050, 24000, 32000, 44100, 48000, 64000, 88200, 96000, 176400, 192000 | Sample rate of input |
| `samplesPerBlock`   | `uint32_t` | 0, 1 ~ 32768 | Number of samples per input block (0 if dynamic) |

> `just_voice_config_t` CANNOT be updated during processing.



#### `just_voice_params_t` (Struct)
| Field | Type | Value | Description |
| ------------------------- | ------- | --------- | - |
| `noiseReductionIntensity` | `float` | 0.f ~ 1.f | Level of noise reduction (If 0, no noise is removed, if 1, noise is completely removed.) |

> `just_voice_params_t` CAN be updated during processing. (It takes 10ms of interpolation time.)



#### Error Codes (Enum)
| Code Name | Value | Description |
| ------------------------- | ------- | --------- |
| `JV_SUCCESS`                           | 0 | No error |
| `JV_NOT_CREATED`                       | 1 | The instance has not been created. |
| `JV_NOT_INITIALIZED`                   | 2 | The instance has not been initialized. (Make sure you have set just_voice_config_t and just_voice_params_t.) |
| `JV_ALREADY_CREATED`                   | 3 | The instance already created. |
| `JV_ALREADY_INITIALIZED`               | 4 | The instance has already been initialized. |
| `JV_NULL_EXCEPTION`                    | 5 | The pointer's value is `NULL`. |
| `JV_ALLOCATION_FAILED`                 | 6 | Internal buffer allocation failed. |
| `JV_NOT_SUPPORTED_NUM_INPUT_CHANNELS`  | 7 | The value of the set number of input channels is outside the supported range. |
| `JV_NOT_SUPPORTED_NUM_OUTPUT_CHANNELS` | 8 | The value of the set number of output channels is outside the supported range. |
| `JV_NOT_SUPPORTED_SAMPLE_RATE`         | 9 | The value of the set sample rate is outside the supported range. |
| `JV_NOT_SUPPORTED_SAMPLES_PER_BLOCK`   | 10 | The value of the set sample per block is outside the supported range. |
| `JV_NOT_SUPPORTED_INTENSITY`           | 11 | The value of the set noise reduction intensity is outside the supported range. |

> Error codes are used as return values from API functions.




### Function
#### `JV_CREATE`
```c
int32_t JV_CREATE(just_voice_handle_t** handle);
```

##### Description
Create an instance. After use, `JV_DESTROY` must be called to release the resource.

| Parameter | Description |
| - | - |
| `handle` | Specifies the pointer to the new instance to be created. The input pointing address must be NULL. |

##### Example
```c
just_voice_handle_t* handle = NULL;
int32_t const create_result = JV_CREATE(&handle);

if (create_result != JV_SUCCESS) { ... }
```



#### `JV_SETUP`
```c
int32_t JV_SETUP(just_voice_handle_t* handle, just_voice_config_t const* config, just_voice_params_t const* params);
```

##### Description
Set the instance from the the values of `just_voice_config_t` and `just_voice_params_t`.

| Parameter | Description |
| - | - |
| `handle` | An instance in Initialized state |
| `config` | A `just_voice_config_t` structure containing settings values. |
| `params` | A `just_voice_params_t` structure containing settings values. |

##### Example
```c
just_voice_handle_t* handle = NULL;
just_voice_config_t const config = { 1, 1, 48000, 480 };
just_voice_params_t const params = { 1.f };

...

int32_t const setup_result = JV_SETUP(handle, &config, &param);

if (setup_result != JV_SUCCESS) { ... }
```



#### `JV_UPDATE`
```c
int32_t JV_UPDATE(just_voice_handle_t* handle, just_voice_params_t const* params);
```

##### Description
Update the instance from the the values of `just_voice_params_t` during processing.

| Parameter | Description |
| - | - |
| `handle` | An instance in Initialized state |
| `params` | A `just_voice_params_t` structure containing settings values. |

##### Example
```c
just_voice_handle_t* handle = NULL;
just_voice_params_t const params = { 1.f };

...

int32_t const update_result = JV_UPDATE(handle, &param);

if (update_result != JV_SUCCESS) { ... }
```



#### `JV_PROCESS`
```c
int32_t JV_PROCESS(just_voice_handle_t* handle, float const* in, float* out, uint32_t length);
```

##### Description
Processes the audio from the input buffer and puts it into the output buffer.
> If `samplesPerBlock` is nonzero, the values of length and `samplesPerBlock` must be equal. Else if is set to 0, it can be used for arbitrary lengths.


| Parameter | Description |
| - | - |
| `handle` | An instance in Initialized state |
| `input` | Input audio buffer |
| `output` | Output audio buffer |
| `length` | The number of input and output samples per channel. |

##### Example
```c
just_voice_handle_t* handle = NULL;
float const* input_buffer = ...;
float* output_buffer = ...;
uint32_t length = ...;

...

int32_t const process_result = JV_PROCESS(handle, input_buffer, output_buffer, length);

if (process_result != JV_SUCCESS) { ... }
```



#### `JV_DESTROY`
```c
int32_t JV_DESTROY(just_voice_handle_t** handle);
```

##### Description
Remove the instance. Releases allocated resources.

| Parameter | Description |
| - | - |
| `handle` | An instance to be removed. After removal, the value `NULL` is set. |

##### Example
```c
just_voice_handle_t* handle = NULL;

...

int32_t destroy_result = JV_DESTROY(&handle);

if (destroy_result != JV_SUCCESS) { ... }
```



#### `JV_GET_LATENCY`
```c
int32_t JV_GET_LATENCY(just_voice_handle_t const* handle, float* latency);
```

##### Description
Calculates and outputs the algorithm latency in ms.

| Parameter | Description |
| - | - |
| `handle` | An instance in Initialized state |
| `latency` | Calculated latency value |

##### Example
```c
just_voice_handle_t* handle = NULL;
float* latency = 0.f;

...

int32_t const latency_result = JV_GET_LATENCY(handle, &latency);

if (latency_result != JV_SUCCESS) { ... }
```



#### `JV_VERSION`
```c
int32_t JV_GET_VERSION(char const** version);
```

##### Description
Check the version number of the SDK.

| Parameter | Description |
| - | - |
| `version` | Version number of this SDK |

##### Example
```c
char const* version;
int32_t const version_result = JV_GET_VERSION(&version);

if (version_result != JV_SUCCESS) { ... }
```
